;---------------------------------------------------------------------------
; Document name: hsi_bproj__define
; Created by:    Andre Csillaghy, May 1999
; Time-stamp: <Fri May 21 2004 13:46:39 csillag soleil.ifi.fh-aargau.ch>
;---------------------------------------------------------------------------
;
;+
; PROJECT:
;       HESSI
;
; NAME:
;       HESSI BACK PROJECTION STRATEGY ABSTRACT CLASS
;
; PURPOSE:
;       This class provides the default implementation for the back
;       projection. It is extended by concrete classe that provide the
;       startegies used to generate the bak projection, now basically the
;       annular sector back projetion or the visibility back projection.
;
; CATEGORY
;       Imaging
;
; CONSTRUCTION:
;       o = HSI_BProj()
;
;       The variable o is the object references used to
;       access back projection maps and associated methods.
;
; INPUTS (CONTROL) PARAMETERS:
;
; OUTPUTS (INFORMATION) PARAMETERS:
;
;
; SOURCE OBJECT:
;       HSI_Modul_Pattern
;
; DESTINATION OBJECTS:
;       HSI_Modpat_Products, HSI_Image
;
; KEYWORDS:
;   On Getdata:
;     THIS_DET_INDEX - return only the map for THIS_DET_INDEX
;                 CLASS_NAME=class_name, $
;       NO_SUM - If set, return pointer array of maps
;     NO_SPATIAL_FREQUENCY_WEIGHT - Don't apply any spatial weighting function
;
;     VRATE - Use this pointer array of vectors instead of calib_eventlist.counts

;
; EXAMPLE:
;
; SEE ALSO:
;       HESSI Utility Reference
;          http://hessi.ssl.berkeley.edu/software/reference.html
;       hsi_modul_pattern__define
;       hsi_bproj_control__defin
;       hsi_bproj_control
;       hsi_bproj_info__define
;
; HISTORY:
;	  1-mar-2011, ras, clean up abandoned pointers
;	  29-apr-2007, Kim.  Use n_elements to check for taper in SET instead of keyword_set
;	  25-apr-2007, ras, remove this_harmonic
;     9-may-2005,ras, trap and pass through NORATE_CORRECT to support
;     back projection without rate correction as required by hsi_pixon_bproj
;
;               21-may-2004--  acs, change the set, last_update to the
;                             new last update counter instead of
;                             systime which does not work on fast
;                             machines. eliminate the set last_update
;                             in ::process as it is already done in framework
;     24-nov-2002, ras, prevent underflows in call to exp() for taper coeffs.
;     Sept 26 2002, ras, added NO_SPATIAL_FREQUENCY_WEIGHT at Metcalf's request
;       Release 6 development: introduction of normal, uniform, taper, and
;                              spatial_frequency_weight keywords
;       Release 3 development, August / September 1999,
;           A Csillaghy, csillag@ssl.berkeley.edu
;       Based on the release 2 software of Richard Schwartz
;       Fundamental developments by Richard Schwartz, GSFC
;     19-dec-2001 added det_index_mask trap to SET, ras
;
;  Modifications
;  28-mar-2017, ras, if cbe_normalize has changed, set update for bproj to force all
;  routines that use bproj to update.  It will force update for bproj itself although the
;  result won't change as cbe_normalize does not affect the bproj
;  3-may-2017, ras, setting this_flatfield to 0, ie using a keyword to pass thru to annsec_bproj instead of the
;  control parameter which could cause reprocessing for other image algs, also using standard block formatting now
;
;  added warning message to using
;  Added Compute_Spatial_Frequency_Weight for added clarity with Uniform, Natural, and Taper options
;  Added Apply_SpFrqWeight to do the summation of the bproj maps with weighting
;-

;----------------------------------------------------------

FUNCTION HSI_Bproj_Strategy::INIT, SOURCE = source, _EXTRA=_extra

  IF NOT Obj_Valid( source )  THEN source = HSI_Modul_Pattern()
  ret=self->Framework::INIT( CONTROL = HSI_BProj_Strategy_control(), $
    info = {im_eband_used: [0.,0.]}, $
    SOURCE=source )

  self->SetSpatialFrequencyWeight

  IF Keyword_Set( _EXTRA ) THEN self->Set, _EXTRA = _extra

  RETURN, ret

END

;----------------------------------------------------------

PRO HSI_Bproj_Strategy::SetSpatialFrequencyWeight

  spatial_frequency_weight = Self->Compute_Spatial_Frequency_Weight()


  self->Framework::Set, SPATIAL_FREQUENCY_WEIGHT = spatial_frequency_weight, $
    /THIS_CLASS_ONLY


END
;--------------------------------------------------------------------
Function HSI_Bproj_Strategy::Compute_Spatial_Frequency_Weight, $
  uniform_weighting = uniform, natural_weighting = natural, $
  taper = taper, $
  _EXTRA = _extra

  grid_param = HSI_Grid_Parameters()
  coll_fwhm = grid_param.pitch /2.
  if exist( uniform ) or exist( natural ) then begin
    case 1 of
      exist( uniform ) : begin
        uniform = uniform >0<1
        natural = 1 - uniform
      end
      else: begin
        natural = natural > 0 < 1
        uniform = 1 - natural
      end
    endcase
  endif else begin
    uniform =  self->Get( /UNIFORM_WEIGHTING )  > 0 < 1
    natural =  self->Get( /NATURAL_WEIGHTING )  > 0 < 1
  endelse
  uniform = 1 - natural
  taper =   exist( taper) ? taper :self->Get( /TAPER )
  taper = Exp( -0.89*((taper/coll_fwhm)^2<50.))

  density =  uniform ? 1. / coll_fwhm : Fltarr( 9 ) + 1.

  spatial_frequency_weight = density*taper

  return, spatial_frequency_weight
END

;----------------------------------------------------------

PRO HSI_Bproj_Strategy::Set, $
  UNIFORM_WEIGHTING = uniform_weighting, $
  NATURAL_WEIGHTING = natural_weighting, $
  TAPER = taper, NOT_FOUND = NOT_found, $
  FLATFIELD = flatfield, $
  DET_INDEX_MASK = det_index_mask, $
  CBE_NORMALIZE = cbe_normalize, $
  _EXTRA = _extra

  set_spat = 0
  IF Keyword_Set(DET_INDEX_MASK) THEN $
    self->Framework::Set, /NEED_UPDATE, /THIS_CLASS_ONLY
  IF Keyword_Set(CBE_NORMALIZE) && cbe_normalize ne Self->Framework::Get(/cbe_normalize) THEN $
    self->Framework::Set, det_index_mask = det_index_mask, /NEED_UPDATE, /THIS_CLASS_ONLY
  IF N_elements( TAPER ) gt 0 THEN BEGIN
    old_taper = self->Get( /TAPER, /THIS_CLASS )
    IF NOT Same_Data( old_taper, taper ) THEN BEGIN
      set_spat = 1
    ENDIF
  ENDIF
  IF N_elements( UNIFORM_WEIGHTING ) gt 0 THEN BEGIN
    old_w = self->Get( /UNIFORM_WEIGHTING, /THIS_CLASS )
    if not set_spat then $
      IF NOT Same_Data( old_w, uniform_weighting ) THEN BEGIN
      set_spat = 1
    ENDIF
    natural_weighting = 1- uniform_weighting
  ENDIF
  IF N_elements( NATURAL_WEIGHTING ) gt 0 THEN BEGIN
    old_w = self->Get( /NATURAL_WEIGHTING, /THIS_CLASS )
    if not set_spat then $
      IF NOT Same_Data( old_w, natural_weighting ) THEN BEGIN
      set_spat = 1
    ENDIF
    uniform_weighting =  1-natural_weighting
  ENDIF
  Self->Framework::Set, Flatfield = flatfield, /no_update ; ras, 05-apr-2017, allways recalculating on getdata
  self->Framework::Set, $
    UNIFORM_WEIGHTING = uniform_weighting, $
    NATURAL_WEIGHTING = natural_weighting, $
    TAPER = taper, NOT_FOUND = NOT_found, $
    DET_INDEX_MASK = det_index_mask, $
    CBE_NORMALIZE  = cbe_normalize, $
    _EXTRA = _extra

  IF set_spat THEN BEGIN
    self->SetSpatialFrequencyWeight
  ENDIF
END
;---------------------
FUNCTION HSI_Bproj_Strategy::Get_det_index_mask, $
  det_index_mask = det_index_mask, $
  this_det_index = this_det_index, $
  no_sum = no_sum

  use_det_mask = keyword_set( det_index_mask ) ? det_index_mask :self->Get( /DET_INDEX_MASK )
  if keyword_set( this_det_index ) then begin
    use_det_mask = bytarr(9)
    use_det_mask[ this_det_index ]++
    no_sum = 1
  endif
  return, use_det_mask
end
;----------------------------------------------------------

FUNCTION HSI_Bproj_Strategy::GetData, $
  CLASS_NAME = class_name, $
  NO_SUM = no_sum, $
  THIS_DET_INDEX = this_det_index, $
  Det_index_mask = det_index_mask,$
  NO_SPATIAL_FREQUENCY_WEIGHT= no_spatial_frequency_weight, $
  cbe_ptr  =  cbe_ptr, $
  map_ptr = map_ptr, $ ;2-jan-2015
  VRATE = vrate, $
  ;NORATE_APPLIES ONLY TO VRATE
  NORATE_CORRECT = NORATE_CORRECT,$
  PTR_OUT = ptr_out, $ ;if set, then return the data in a pointer
  this_flatfield = this_flatfield, $

  _EXTRA = _extra

  IF Keyword_Set( _EXTRA ) THEN self->Set, _EXTRA = _extra
  Flatfield = Self->Get( /FLATFIELD )
  if exist( this_flatfield ) then flatfield = this_flatfield
  IF N_Elements( class_name ) THEN IF class_name NE Obj_Class( self ) THEN BEGIN
    RETURN, self->Framework::GetData( CLASS_NAME = class_name, $
      THIS_DET_INDEX = this_det_index, $
      flatfield = flatfield, $
      _EXTRA = _extra )
  ENDIF
  det_index_mask = self->Get_det_index_mask( det_index_mask = det_index_mask, $
    THIS_DET_INDEX = this_det_index, no_sum = this_no_sum )
  default, no_sum, 0
  default, ptr_out, 0

  this_no_sum = exist( this_no_sum ) ? 1 : no_sum
  ;In both cases, return all 9 bproj mats in fltarr( nx, ny, 9 ) and do weighted sum using apply_spfrqweight
  if self->need_update() then $
    data=self->Framework::GetData( _EXTRA = _extra) else begin
    vrate = ptr_chk( vrate ) ? vrate : Self->GetData( class='hsi_calib_eventlist' )

    data = Self->hsi_bproj_alg_hook( data_ptr=vrate,$
      /no_sum, $
      /no_spatial_frequency_weight, $
      /no_ptr, $      
      flatfield = flatfield, $
      cbe_ptr =  cbe_ptr, $
      map_ptr = map_ptr, $ ;2-jan-2015
      NORATE_CORRECT = norate_correct,$
      _EXTRA = _extra )
  endelse
  ;Here is where the spatial weighting is applied and the images summed or left separate
  data = Self->Apply_SpFrqWeight( data, det_index_mask=det_index_mask, no_sum=this_no_sum,$
    NO_SPATIAL_FREQUENCY_WEIGHT= no_spatial_frequency_weight )
  data = ptr_out ? ptr_new( data ) : data
  return, data
END
;----------------------------------------------------------
; This function is no longer needed. 2-may-2017, ras, this was
; an attempt to halt reprocessing when using the machinery of bproj but
; turns out to be unnecessary
FUNCTION HSI_Bproj_Strategy::Use_Vrate, vrate,  _EXTRA = _extra
  return, self->getdata( vrate = vrate, _extra = _extra)
END
;-----------------------------------------------------------------

FUNCTION HSI_Bproj_Strategy::Normalize_for_Det_Mask, $
  input,  det_index_mask
  ;Normalize the spatial weighting to valid detectors
  test = n_elements( input ) eq n_elements( det_index_mask )
  case test of
    1: begin
      result = input * det_index_mask
      result /= total( result )
    end
    else: result = input / total( input )
  endcase

  return, result
end

;--------------------------------------------------
FUNCTION HSI_Bproj_Strategy::Apply_SpFrqWeight, $
  image, $
  SPATIAL_FREQUENCY_WEIGHT = spatial_frequency_weight, $
  NO_SPATIAL_FREQUENCY_WEIGHT= no_SpFrqWeight, $
  det_index_mask = det_index_mask, $
  no_sum = no_sum, $
  _EXTRA = _extra

  default, det_index_mask, self->Get( /det_index_mask ) ;ptr_valid( image_ptr )
  ndet = total( det_index_mask, /int )
  
  image_dim = size( /dimension, image )
  nimage = last_item( image_dim )
  ; image_dim should be [nx, ny, nimage] or [ nx*ny, nimage ]
  ; 
  ; As this may be used with harmonic visibilities in the future, we don't assume ndet and nimage are 9
  ;if ndet  ne nimage then message,'NIMAGE and NDET should be equal!! NIMAGE='+strtrim(nimage,2)+' NDET='+strtrim(ndet,2)
  default, no_sum, 0
  if ptr_chk( image ) then message,'image_ptr is a pointer, should be deprecated'
  spatial_frequency_weight = Self->Compute_Spatial_Frequency_Weight( _extra = _extra )
  case 1 of
    Keyword_set( NO_SpFrqWeight ) : SpFrqWeight = fltarr(nimage)+1.0
    Keyword_set( spatial_frequency_weight ) : SpFrqWeight = spatial_frequency_weight
    else: SpFrqWeight = self->Get( /SPATIAL_FREQUENCY_WEIGHT )
  endcase
  ;Normalize to valid detectors
  sfw_arr =  Self->Normalize_for_Det_Mask( SpFrqWeight, det_index_mask )
  z = where( det_index_mask, nz )
  if n_elements( sfw_arr ) gt nz then sfw_arr = sfw_arr[z]
  if nimage gt nz then begin
    temp = fltarr( nimage )
    temp[z] = sfw_arr
    sfw_arr = temp
  endif

  image = stack_weight( image, sfw_arr, sum = 1 - no_sum )  ;weight the images in the stack and most often sum them
  RETURN, image
END

;-----------------------------------------------------------------

PRO Hsi_Bproj_Strategy::Process, _EXTRA = _extra
  flatfield = Self->Get(/flatfield)
  bp = self->HSI_Bproj_Alg_Hook( _EXTRA =  _extra, flatfield=flatfield,  /no_ptr, /no_det_mask, /no_spatial )

  old_bp = Self->framework::getdata(/noprocess)
  if ptr_chk(old_bp) then heap_free, old_bp

  self->Framework::SetData, bp
  self->Set, NEED_UPDATE = 0

  self->set, im_eband_used = self->get( /energy_band )

  ; acs 2004-05-21 this shoudl not be necessary as it is done in setdata
  ; self->Set, LAST_UPDATE = Systime(/SECONDS)
  ;
  ;    self->Framework::Set, LAST_UPDATE = !hsi_last_update_counter
  ;    !hsi_last_update_counter = !hsi_last_update_counter + 1

END

;----------------------------------------------------------

PRO HSI_Bproj_Strategy__Define

  bproj = {HSI_Bproj_Strategy, $
    INHERITS Framework}

END


;---------------------------------------------------------------------------
; End of 'hsi_bproj__define.pro'.
;---------------------------------------------------------------------------
