;+
; PROJECT:
;   HESSI
; NAME:
;   HSI_CART_BPROJ
;
; PURPOSE:
;   Returns the summed back-projection of the hessi calibrated eventlist.
;   Optimized for speed. Another version for individual bproj maps for each  sc will come later
;
; CATEGORY:
;   HESSI, UTIL
;
; CALLING SEQUENCE:
;   hsi_cart_modul_pattern, obj, cc, cmpat
;   map=hsi_cart_bproj( cc, cmpat, data, $
;      no_tot_vrate = no_tot_vrate,$
;      square = square, $ ;reform to a square dim
;      flatfield = flatfield, $
;      nphz = nphz, $
;      _extra = _extra)
;
;
; CALLS:
;   none
;
; INPUTS:
;      cc - extended and concatenated calibrated eventlist built for the current
;       det_index_mask built from hsi_cart_modul_pattern
;      cmpat - modulation pattern structure from hsi_cart_modul_pattern
; OPTIONAL INPUTS:
;      data - vector of same length as cc, replaces count field from calib eventlist
;
; OUTPUTS:
;       Returns the map as a pointer (if SUM eq 0) or as a 2-d image in polar coordinates.
;
; OPTIONAL OUTPUTS:
;   none
;
; KEYWORDS:
;     no_tot_vrate - don't add the dc component, usually the total of the data vector
;     square -   not implemented
;     flatfield - not implemented yet
; COMMON BLOCKS:
;   none
;
; SIDE EFFECTS:
;   none
;
; RESTRICTIONS:
;   none
;
; PROCEDURE:
;   none
;
; MODIFICATION HISTORY:
;   Version 1, richard.schwartz@gsfc.nasa.gov, 20-aug-2015, based on hsi_annsec_bproj computational optimizations
;   24-may-2017, use_rate is an explicit keyword to pass thru to hsi_cart_bproj_rates
;
;-

;prep the data for rate based back-projection
function hsi_cart_bproj_struct, cc, cmpat, data, $
  no_tot_vrate = no_tot_vrate, $
  counts_summed = counts_summed, $
  ;square = square, $ ;reform to a square dim
  nphz = nphz, $
  csmap1 = csmap1, $
  use_rate = use_rate, $
  _extra = _extra

if ~is_number( data[0] ) then data = cc.count
vrate = is_struct( data ) ? data.count : data

default, no_tot_vrate, 0
default, use_rate, 1
nphz = cmpat.nphz

;prep the data for rate based back-projection
hsi_cart_bproj_rates, cc, cmpat, vrate, use_rate, tot_vrate

  cvrate = vrate * (*cmpat.macosphz) & svrate =  vrate * (*cmpat.masinphz)
  ncos   = n_elements( cc ) / nphz 
  summer = make_array( nphz, value=1.0)
  cvrate = reform( summer # reform( cvrate, nphz, ncos) )
  svrate = reform( summer # reform( svrate, nphz, ncos) )
  
  cvrate2 = fltarr( ncos / 2 )
  svrate2 = cvrate2
  nmap = cmpat.nmap
  dmap = (nmap[1:*]-nmap ) / 2
  ;The two sets of coefficients are from 0 to pi and pi to 2pi. The sign difference comes from
  ; cos(a) = cos(-a) and sin(a) = -sin(-a)
  ;The phase can go through many cycles from 0 to pi but reverses sign at pi so by construction we have
  ; cos(-a) = cos(a) and sin(-a) = sin(a)
  for i=0,cmpat.nmaps - 1 do cvrate2[ nmap[i]/2 ] = cvrate[ nmap[i]:nmap[i]+dmap[i]-1 ] + cvrate[nmap[i]+dmap[i]:nmap[i+1]-1]
  for i=0,cmpat.nmaps - 1 do svrate2[ nmap[i]/2 ] = svrate[ nmap[i]:nmap[i]+dmap[i]-1 ] - svrate[nmap[i]+dmap[i]:nmap[i+1]-1]
  ;bp = cvrate2 # *cmpat.cos_map1 - svrate2 # *cmpat.sin_map1
  ;the minus sign here is from the law of cosines, cos( a + b ) = cos(a)cos(b) - sin(a)sin(b)
  
  bp = ~exist( csmap1 ) ?  matrix_multiply( cvrate2, *cmpat.cos_map1) - matrix_multiply( svrate2, *cmpat.sin_map1 ) : $
    matrix_multiply( cvrate2, *csmap1.cos_map1) - matrix_multiply( svrate2, *csmap1.sin_map1 ) 


 bp = ~no_tot_vrate ? bp + tot_vrate : bp
 counts_summed = tot_vrate
 return, bp
 end
