
;+
; PROJECT:
;   HESSI
; NAME:
;   HSI_CART_PROFILE
;
; PURPOSE:
;   Returns time profile for an input image as expected counts
;   per time interval of associated calbrated eventlist.
;
; CATEGORY:
;   HESSI, UTIL
;
;
; CALLING SEQUENCE:
;
;   hsi_cart_modul_pattern, obj, cc, cmpat
;   map=hsi_cart_profile(image, cc, cmpat )
;

; CALLS:
;   none
;
; INPUTS:
;   Image  - Image to make profile from.  
;   cc - extended and concatenated calibrated eventlist built for the current
;       det_index_mask built from hsi_cart_modul_pattern
;   cmpat - modulation pattern structure from hsi_cart_modul_pattern
;    IDL> help, cc, cmpat,/st
;    ** Structure HSI_CALIB_EVENT_STACK_PLUS00, 16 tags, length=60, data length=59:
;    ROLL_ANGLE      FLOAT         0.0490874
;    MODAMP          FLOAT           1.11610
;    PHASE_MAP_CTR   FLOAT          0.258312
;    GRIDTRAN        FLOAT          0.144507
;    FLUX_VAR        FLOAT          0.895600
;    BACKGROUND      FLOAT          0.000000
;    COUNT           FLOAT           3.00000
;    LIVETIME        FLOAT           50.1348
;    GAP             BYTE         0
;    ATTEN_STATE     INT              1
;    TIME_BIN        FLOAT       0.000976563
;    VRATE0          FLOAT        0.00617640
;    CVRATE          FLOAT        0.00666481
;    SVRATE          FLOAT        0.00176094
;    MACOSPHZ        FLOAT           1.07908
;    MASINPHZ        FLOAT          0.285108
;    ** Structure HSI_CART_MODUL_PATTERN_STR, 18 tags, length=132, data length=131:
;    COS_MAP0        POINTER   <PtrHeapVar48979>
;    COS_MAP1        POINTER   <PtrHeapVar48980>
;    SIN_MAP0        POINTER   <PtrHeapVar48981>
;    SIN_MAP1        POINTER   <PtrHeapVar48982>
;    MACOSPHZ        POINTER   <PtrHeapVar48983>
;    MASINPHZ        POINTER   <PtrHeapVar48984>
;    IMAGE_DIM       INT       Array[2]
;    PIXEL_SIZE      FLOAT     Array[2]
;    PIXEL_SCALE     FLOAT           1.00000
;    NMAPS           LONG                 6
;    NMAP            LONG      Array[7]
;    NPHZ            LONG                12
;    DET_INDEX_MASK  BYTE      Array[9]
;    HARMONIC        INT              1
;    DET_EFF_REL     FLOAT     Array[6]
;    DET_EFF_AVG     FLOAT          0.548151
;    ENERGY_BAND     FLOAT     Array[2]
;    XYOFFSET        FLOAT     Array[2]
;    ;
; OPTIONAL INPUTS:
;
;
; OUTPUTS:
;       none explicit, only through commons;
;
; OPTIONAL OUTPUTS:
;   none
;
; KEYWORDS:
; COMMON BLOCKS:
;   none
;
; SIDE EFFECTS:
;   none
;
; RESTRICTIONS:
;   none
;
; PROCEDURE:
;   The basic procedure is to multiply the image by the modulation pattern pixel by pixel
;   and sum for each time interval.  Here, the modulation pattern is decomposed into
;   cosine and sine maps and combined according to the relative phase. This facilitates
;   reuse of the modulation patterns from the first pi rotation and the reuse of some
;   patterns within that group.  The reuse of the modulation patterns holds the number
;   of arithmetic operations to a minimum.
;
; MODIFICATION HISTORY:
;   20-aug-2015, richard.schwartz, built on cos and sin modpats on a Cartesian grid
;   modeled after hsi_annsec_profile, designed for fast processing so this is not the flexible
;   interface yet
;   26-may-2017, richard.schwartz@nasa.gov, get vrate0, cvrate, and svrate from cc or cmpat
;   this is particularly useful when computing the profile for one grid at a time
;   12-jun-2017, ras, fixed sign error in "out    +=  out_cos * cvrate - out_sin * svrate' had been "+ out_sin" but that's
;   not the cos( a+b ), ie. cos(a+b) = cos(a)cos(b) - sin(a)sin(b) and cvrate and svrate contain cos and sin of the phase_map_ctr!


;-
function hsi_cart_profile, image, cc, cmpat
  
;the vrate0, svrate, and cvrate are the collected coefficents to use with the product of the image and cos/sin modulation patterns
  ;temp.time_bin = time_bins[i]
  ;temp.vrate0 = temp.livetime * temp.gridtran * temp.flux_var * det_eff_rel[i] * time_bins[i] - average transmission coefficient
  ;temp.macosphz = cos( temp.phase_map_ctr ) * temp.modamp * (1 - temp.gap) * weight[ sel[i] ] ;used in bproj
  ;temp.masinphz = sin( temp.phase_map_ctr ) * temp.modamp * (1 - temp.gap) * weight[ sel[i] ]
  ;temp.cvrate = temp.vrate0 * temp.macosphz - avg trans coefficient with cos of the phase offset from 0 point
  ;temp.svrate = temp.vrate0 * temp.masinphz
  
  
  case 1 of
    have_tag( cc, 'vrate0') :begin
      vrate0 = cc.vrate0
      svrate = cc.svrate
      cvrate = cc.cvrate
      end
      else: begin
        test = have_tag( cmpat, 'vrate0')
        if ~test then message, 'No access to vrate0, svrate, cvrate, see hsi_cart_modul_pattern '
        vrate0 = *cmpat.vrate0
        svrate = *cmpat.svrate
        cvrate = *cmpat.cvrate
end
  endcase
  
  out = total(image) * vrate0
  ; cos_map0 and sin map0 are the cos and sin of the phase maps for each pixel in the fov, they vary from -1 to 1
  ; We only need to multiply the image by these maps, then use the phase diff coefficients to compute
  ; the time profiles. And then scale those for grid_tran, modamp, flux_var, and livetime to get the final
  ; expectation values. cvrate and svrate holds those products and there is a cvrate and svrate for 
  ; each bin in the calibrated eventlist
  out_cos = reform( matrix_multiply( image[*], *cmpat.cos_map0 ) )
  out_sin = reform( matrix_multiply( image[*], *cmpat.sin_map0 ))
  
  ;expand by nphz for phase, separate, and then by two for !pi to 2*!pi
  out_cos = (( fltarr( cmpat.nphz ) + 1.0 ) # out_cos )[*]
  out_cos =  [ out_cos, out_cos ]
  out_sin = (( fltarr( cmpat.nphz ) + 1.0 ) # out_sin )[*]
  out_sin =  [ out_sin, -out_sin]
  
  out    +=  out_cos * cvrate - out_sin * svrate
return, out
end
